import { state, dom, REPEAT_DELAY, INITIAL_REPEAT_DELAY } from './state.js';
import * as ui from './ui.js';
import * as api from './api.js';

export function setupInputHandlers() {
    document.addEventListener('keydown', handleKeyDown);
    requestAnimationFrame(pollGamepads);
}

function dispatchKeyEvent(keyName) {
    const blockAllInput = dom.progressOverlay && !dom.progressOverlay.classList.contains('hidden') && dom.progressCompletionButtons?.classList.contains('hidden');
    if (blockAllInput && !['Escape', 'B', 'Backspace'].includes(keyName)) {
        return;
    }
    document.dispatchEvent(new KeyboardEvent('keydown', { key: keyName, code: keyName, bubbles: true }));
}

function pollGamepads() {
    if (state.isEmuAppRunning) {
        requestAnimationFrame(pollGamepads);
        return;
    }
    if (!navigator.getGamepads) {
        requestAnimationFrame(pollGamepads);
        return;
    }
    const gamepad = navigator.getGamepads()[0];
    if (!gamepad) {
        requestAnimationFrame(pollGamepads);
        return;
    }
    
    if (state.isVirtualKeyboardActive && document.activeElement === dom.virtualKeyboardInput) {
        const anyButtonPressed = gamepad.buttons.some(button => button.pressed);
        const h = gamepad.axes[0] || 0;
        const v = gamepad.axes[1] || 0;
        const thresh = 0.5;
        const anyDirectionalInput = Math.abs(h) > thresh || Math.abs(v) > thresh;

        if (anyButtonPressed || anyDirectionalInput) {
            ui.updateKeyboardButtonSelection();
            requestAnimationFrame(pollGamepads);
            return;
        }
    }
    
    const now = Date.now();
    const checkButton = (buttonIndex, stateKey, action) => {
        if (gamepad.buttons[buttonIndex]?.pressed && !state.previousButtonStates[stateKey]) {
            action();
        }
        state.previousButtonStates[stateKey] = gamepad.buttons[buttonIndex]?.pressed;
    };
    const processDirection = (buttonOrAxisActive, stateKey, eventKey) => {
        const blockDirectionalInput = (dom.progressOverlay && !dom.progressOverlay.classList.contains('hidden') && dom.progressCompletionButtons?.classList.contains('hidden'));
        if (buttonOrAxisActive && !blockDirectionalInput) {
            if (!state.previousButtonStates[stateKey]) {
                dispatchKeyEvent(eventKey);
                state.previousButtonStates[stateKey] = true;
                state.buttonPressedTime[stateKey] = now;
                state.lastDirectionalInputTime[stateKey] = now;
            } else if (now - state.buttonPressedTime[stateKey] >= INITIAL_REPEAT_DELAY) {
                if (now - state.lastDirectionalInputTime[stateKey] >= REPEAT_DELAY) {
                    dispatchKeyEvent(eventKey);
                    state.lastDirectionalInputTime[stateKey] = now;
                }
            }
        } else {
            state.previousButtonStates[stateKey] = false;
            state.buttonPressedTime[stateKey] = 0;
        }
    };

    const isModalOrPopupActive = () =>
        state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive ||
        state.isVersionMenuActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive || state.isSavePromptActive || state.isPowerMenuActive;

    const canToggleTabMenu = () =>
        !state.isInitialSelectionActive && !state.isEmuAppRunning && !isModalOrPopupActive();

    checkButton(0, 'A', () => dispatchKeyEvent('Enter'));
    checkButton(1, 'B', () => dispatchKeyEvent('Backspace'));
    checkButton(2, 'X', () => { if (state.navigationMode === 'gameList' && !isModalOrPopupActive()) api.toggleFavorite(); });
    checkButton(3, 'Y', () => { if (canToggleTabMenu()) { if (state.isTabMenuActive) ui.closeTabMenu(); else ui.openTabMenu(); } });
    checkButton(8, 'BackSelect', () => { if (canToggleTabMenu()) api.showOnlineLobby(); });
    checkButton(9, 'Start', () => { if (canToggleTabMenu()) { if (state.isTabMenuActive) ui.closeTabMenu(); else ui.openTabMenu(); } });

    const h = gamepad.axes[0] || 0;
    const v = gamepad.axes[1] || 0;
    const thresh = 0.5;

    processDirection(gamepad.buttons[12]?.pressed || v < -thresh, 'DPadUp', 'ArrowUp');
    processDirection(gamepad.buttons[13]?.pressed || v > thresh, 'DPadDown', 'ArrowDown');
    processDirection(gamepad.buttons[14]?.pressed || h < -thresh, 'DPadLeft', 'ArrowLeft');
    processDirection(gamepad.buttons[15]?.pressed || h > thresh, 'DPadRight', 'ArrowRight');

    requestAnimationFrame(pollGamepads);
}

function handleBackButtonLogic() {
    if (!dom.progressCompletionButtons?.classList.contains('hidden')) {
        dom.progressAcceptButton?.click();
        return;
    }
    if (state.isInitialSelectionActive) return;
    if (state.isVirtualKeyboardActive) { ui.closeVirtualKeyboard(); return; }
    if (state.isPowerMenuActive) { ui.closePowerMenu(); return; }
    if (state.isSavePromptActive) { ui.hideSavePrompt(); api.savePromptResponse('no'); return; }
    if (state.isLobbyMenuActive) { ui.closeLobbyMenu(); return; }
    if (state.isPlayDownloadMenuActive) { ui.closePlayDownloadMenu(); return; }
    if (state.isSaveManageMenuActive) { ui.closeSaveManageMenu(); return; }
    if (state.isVersionMenuActive) { ui.closeVersionMenu(); return; }
    if (state.navigationMode === 'tabConsoleList') { ui.hideTabConsoleList(); return; }
    if (state.isTabMenuActive) { ui.closeTabMenu(); return; }
    if (state.isShowingFavorites || state.isShowingDownloaded || dom.gameSearchInput.value) {
        dom.gameSearchInput.value = '';
        api.loadGamesFromConsole(state.selectedConsoleName);
        return;
    }
    if (state.navigationMode === 'letterButtons') {
        state.navigationMode = 'gameList';
        dom.letterJumpColumn.classList.remove('column-focused');
        ui.setActiveLetterButton(null);
        if (state.selectedIndex !== -1 && dom.gameListUl.children[state.selectedIndex]) {
            dom.gameListUl.children[state.selectedIndex].focus();
        }
        return;
    }
    ui.openTabMenu();
}

function handleKeyDown(event) {
    if (state.isEmuAppRunning) {
        if (event.key === 'Escape' || event.key === 'Backspace') {
            event.preventDefault();
            api.killEmuApp();
        }
        return;
    }

    const activeElementIsInput = document.activeElement.tagName === 'INPUT';
    if (event.key === 'Backspace' && activeElementIsInput) return;

    if (event.key === 'Escape' || event.key === 'Backspace') {
        event.preventDefault();
        if (dom.progressOverlay && !dom.progressOverlay.classList.contains('hidden') && dom.progressCompletionButtons?.classList.contains('hidden')) {
            api.killEmuApp();
        } else if (state.isSupportPopupActive) {
            dom.supportPopupCloseButton.click();
        } else {
            handleBackButtonLogic();
        }
        return;
    }

    const isModalOrPopupActive = () =>
        state.isSupportPopupActive || state.isPlayDownloadMenuActive || state.isSaveManageMenuActive ||
        state.isVersionMenuActive || state.isLobbyMenuActive || state.isVirtualKeyboardActive || state.isSavePromptActive || state.isPowerMenuActive;

    const blockInput = (
        (!dom.progressCompletionButtons?.classList.contains('hidden')) ||
        (dom.progressOverlay && !dom.progressOverlay.classList.contains('hidden') && dom.progressCompletionButtons?.classList.contains('hidden'))
    );

    if (event.key === 'Tab') {
        event.preventDefault();
        if (!state.isInitialSelectionActive && !state.isEmuAppRunning && !isModalOrPopupActive()) {
            if (state.isTabMenuActive) {
                if (!state.isVirtualKeyboardActive && state.navigationMode !== 'tabConsoleList') {
                    ui.closeTabMenu();
                }
            } else {
                ui.openTabMenu();
            }
        }
        return;
    }

    if (state.isVirtualKeyboardActive) { handleVirtualKeyboardKeys(event); return; }
    if (state.isPowerMenuActive) { handlePowerMenuKeys(event); return; }
    if (!dom.progressCompletionButtons?.classList.contains('hidden')) { handleProgressCompletionKeys(event); return; }
    if (state.isSupportPopupActive) { handleSupportPopupKeys(event); return; }
    if (state.isPlayDownloadMenuActive) { handlePlayDownloadMenuKeys(event); return; }
    if (state.isSaveManageMenuActive) { handleSaveManageMenuKeys(event); return; }
    if (state.isVersionMenuActive) { handleVersionMenuKeys(event); return; }
    if (state.isTabMenuActive) { handleTabMenuKeys(event); return; }
    if (state.isSavePromptActive) { handleSavePromptKeys(event); return; }
    if (state.isLobbyMenuActive) { handleLobbyMenuKeys(event); return; }
    if (state.isInitialSelectionActive) { handleInitialConsoleListKeys(event); return; }

    if (!blockInput && !state.isTabMenuActive && !state.isInitialSelectionActive) {
        switch (state.navigationMode) {
            case 'gameList': handleGameListKeys(event); break;
            case 'letterButtons': handleLetterButtonKeys(event); break;
        }
    }
}

function handleInitialConsoleListKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            state.initialConsoleListItemIndex = (state.initialConsoleListItemIndex - 1 + state.initialConsoleListItems.length) % state.initialConsoleListItems.length;
            ui.updateInitialConsoleListSelection();
            break;
        case 'ArrowDown':
            state.initialConsoleListItemIndex = (state.initialConsoleListItemIndex + 1) % state.initialConsoleListItems.length;
            ui.updateInitialConsoleListSelection();
            break;
        case 'Enter':
            if (state.initialConsoleListItemIndex !== -1) state.initialConsoleListItems[state.initialConsoleListItemIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleGameListKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            const prevIndex = (state.selectedIndex - 1 + state.currentGames.length) % state.currentGames.length;
            ui.selectGame(prevIndex, true, true);
            break;
        case 'ArrowDown':
            const nextIndex = (state.selectedIndex + 1) % state.currentGames.length;
            ui.selectGame(nextIndex, true, true);
            break;
        case 'ArrowLeft':
            state.navigationMode = 'letterButtons';
            dom.letterJumpColumn.classList.add('column-focused');
            state.letterButtonIndex = Array.from(dom.letterButtons).indexOf(state.activeLetterButton)
            if (state.letterButtonIndex === -1 && dom.letterButtons.length > 0) state.letterButtonIndex = 0;
            ui.setActiveLetterButton(dom.letterButtons[state.letterButtonIndex]);
            break;
        case 'ArrowRight':
            if (state.selectedIndex !== -1 && state.currentGames[state.selectedIndex]?.versions.length > 1) {
                ui.openVersionMenu(state.currentGames[state.selectedIndex]);
            }
            break;
        case 'Enter':
            if (state.selectedIndex !== -1) ui.handleGameSelectionAction(state.currentGames[state.selectedIndex]);
            break;
        case 'F1':
            api.toggleFavorite();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleLetterButtonKeys(event) {
    let jumped = false;
    switch (event.key) {
        case 'ArrowUp':
            state.letterButtonIndex = (state.letterButtonIndex - 1 + dom.letterButtons.length) % dom.letterButtons.length;
            jumped = true;
            break;
        case 'ArrowDown':
            state.letterButtonIndex = (state.letterButtonIndex + 1) % dom.letterButtons.length;
            jumped = true;
            break;
        case 'ArrowRight':
            state.navigationMode = 'gameList';
            dom.letterJumpColumn.classList.remove('column-focused');
            ui.setActiveLetterButton(null);
            if (state.selectedIndex !== -1) dom.gameListUl.children[state.selectedIndex]?.focus();
            break;
        case 'Enter':
            state.navigationMode = 'gameList';
            dom.letterJumpColumn.classList.remove('column-focused');
            if (state.selectedIndex !== -1) dom.gameListUl.children[state.selectedIndex]?.focus();
            break;
        default: return;
    }
    if (jumped) {
        const button = dom.letterButtons[state.letterButtonIndex];
        ui.setActiveLetterButton(button);
        const letter = button.dataset.letter;
        const foundIndex = state.currentGames.findIndex(g => {
            let title = g.baseName.toUpperCase();
            if (state.selectedConsoleName.startsWith('Arcade') && state.arcadeTitles && state.arcadeTitles[g.baseName]) {
                title = state.arcadeTitles[g.baseName].toUpperCase();
            }
            return letter === '#' ? !/^[A-Z]/.test(title) : title.startsWith(letter);
        });
        if (foundIndex !== -1) ui.selectGame(foundIndex, false, true);
    }
    event.preventDefault();
}

function handleVersionMenuKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            state.versionButtonIndex = (state.versionButtonIndex - 1 + state.versionButtons.length) % state.versionButtons.length;
            ui.updateVersionButtonSelection();
            break;
        case 'ArrowDown':
            state.versionButtonIndex = (state.versionButtonIndex + 1) % state.versionButtons.length;
            ui.updateVersionButtonSelection();
            break;
        case 'Enter':
            if (state.versionButtonIndex !== -1) state.versionButtons[state.versionButtonIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleTabMenuKeys(event) {
    if (state.navigationMode === 'tabConsoleList') {
        handleTabConsoleListKeys(event);
        return;
    }

    const currentButton = state.tabMenuButtons[state.tabMenuButtonIndex];
    const isRelayConfigSelected = currentButton === dom.relayConfigButton;

    if (isRelayConfigSelected) {
        if (event.key === 'ArrowLeft') {
            state.relay.currentIndex = (state.relay.currentIndex - 1 + state.relay.options.length) % state.relay.options.length;
            ui.updateRelayConfigDisplay();
            event.preventDefault();
            return;
        }
        if (event.key === 'ArrowRight') {
            state.relay.currentIndex = (state.relay.currentIndex + 1) % state.relay.options.length;
            ui.updateRelayConfigDisplay();
            event.preventDefault();
            return;
        }
    }

    switch (event.key) {
        case 'ArrowUp':
            state.tabMenuButtonIndex = (state.tabMenuButtonIndex - 1 + state.tabMenuButtons.length) % state.tabMenuButtons.length;
            ui.updateTabMenuButtonSelection();
            break;
        case 'ArrowDown':
            state.tabMenuButtonIndex = (state.tabMenuButtonIndex + 1) % state.tabMenuButtons.length;
            ui.updateTabMenuButtonSelection();
            break;
        case 'Enter':
            if (state.tabMenuButtonIndex !== -1 && !isRelayConfigSelected) {
                state.tabMenuButtons[state.tabMenuButtonIndex]?.click();
            }
            break;
        default: return;
    }
    event.preventDefault();
}

function handleTabConsoleListKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            state.tabConsoleListItemIndex = (state.tabConsoleListItemIndex - 1 + state.tabConsoleListItems.length) % state.tabConsoleListItems.length;
            ui.updateTabConsoleListSelection();
            break;
        case 'ArrowDown':
            state.tabConsoleListItemIndex = (state.tabConsoleListItemIndex + 1) % state.tabConsoleListItems.length;
            ui.updateTabConsoleListSelection();
            break;
        case 'Enter':
            if (state.tabConsoleListItemIndex !== -1) state.tabConsoleListItems[state.tabConsoleListItemIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleVirtualKeyboardKeys(event) {
    if (document.activeElement === dom.virtualKeyboardInput) return;

    const flatGrid = state.keyboardGrid.flat();
    if (flatGrid.length === 0 || state.keyboardButtonIndex < 0) return;

    const currentKey = flatGrid[state.keyboardButtonIndex];
    let nextIndex = state.keyboardButtonIndex;

    const getColumnData = (keyInfo) => {
        let colStart = 0;
        const row = state.keyboardGrid[keyInfo.row];
        for (let i = 0; i < keyInfo.col; i++) {
            colStart += row[i].span;
        }
        const center = colStart + (keyInfo.span / 2);
        return { start: colStart, center: center };
    };

    switch (event.key) {
        case 'ArrowLeft':
            if (state.keyboardButtonIndex > 0) nextIndex--;
            else nextIndex = flatGrid.length - 1;
            break;
        case 'ArrowRight':
            if (state.keyboardButtonIndex < flatGrid.length - 1) nextIndex++;
            else nextIndex = 0;
            break;
        case 'ArrowUp':
        case 'ArrowDown': {
            const direction = event.key === 'ArrowUp' ? -1 : 1;
            const targetRowIndex = currentKey.row + direction;
            if (targetRowIndex >= 0 && targetRowIndex < state.keyboardGrid.length) {
                const targetRow = state.keyboardGrid[targetRowIndex];
                const { center: currentKeyCenter } = getColumnData(currentKey);
                let bestMatch = targetRow[0];
                let minDistance = Infinity;
                for (const keyInTargetRow of targetRow) {
                    const { center: targetKeyCenter } = getColumnData(keyInTargetRow);
                    const distance = Math.abs(currentKeyCenter - targetKeyCenter);
                    if (distance < minDistance) {
                        minDistance = distance;
                        bestMatch = keyInTargetRow;
                    }
                }
                nextIndex = flatGrid.indexOf(bestMatch);
            }
            break;
        }
        case 'Enter':
            currentKey.button.click();
            break;
        default: return;
    }

    if (nextIndex !== state.keyboardButtonIndex) {
        state.keyboardButtonIndex = nextIndex;
        ui.updateKeyboardButtonSelection();
    }
    event.preventDefault();
}

function handleSupportPopupKeys(event) {
    switch (event.key) {
        case 'ArrowLeft':
        case 'ArrowRight':
            state.supportPopupButtonIndex = (state.supportPopupButtonIndex + 1) % state.supportPopupButtons.length;
            ui.updateSupportPopupButtonSelection();
            break;
        case 'Enter':
            if (state.supportPopupButtonIndex !== -1) state.supportPopupButtons[state.supportPopupButtonIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handlePlayDownloadMenuKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            state.playDownloadMenuIndex = (state.playDownloadMenuIndex - 1 + state.playDownloadMenuButtons.length) % state.playDownloadMenuButtons.length;
            ui.updatePlayDownloadMenuSelection();
            break;
        case 'ArrowDown':
            state.playDownloadMenuIndex = (state.playDownloadMenuIndex + 1) % state.playDownloadMenuButtons.length;
            ui.updatePlayDownloadMenuSelection();
            break;
        case 'Enter':
            if (state.playDownloadMenuIndex !== -1) state.playDownloadMenuButtons[state.playDownloadMenuIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleSaveManageMenuKeys(event) {
    switch (event.key) {
        case 'ArrowUp':
            state.saveManageMenuIndex = (state.saveManageMenuIndex - 1 + state.saveManageMenuButtons.length) % state.saveManageMenuButtons.length;
            ui.updateSaveManageMenuSelection();
            break;
        case 'ArrowDown':
            state.saveManageMenuIndex = (state.saveManageMenuIndex + 1) % state.saveManageMenuButtons.length;
            ui.updateSaveManageMenuSelection();
            break;
        case 'Enter':
            if (state.saveManageMenuIndex !== -1) state.saveManageMenuButtons[state.saveManageMenuIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleProgressCompletionKeys(event) {
    const visibleButtons = [dom.progressAcceptButton, dom.progressPlayNowButton].filter(btn => !btn.classList.contains('hidden'));
    if (visibleButtons.length === 0) return;
    let currentIndex = visibleButtons.findIndex(btn => document.activeElement === btn);
    if (currentIndex === -1) currentIndex = 0;
    
    switch (event.key) {
        case 'ArrowLeft':
        case 'ArrowRight':
            currentIndex = (currentIndex + 1) % visibleButtons.length;
            visibleButtons[currentIndex]?.focus();
            break;
        case 'Enter':
            document.activeElement?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handleSavePromptKeys(event) {
    switch (event.key) {
        case 'ArrowLeft':
        case 'ArrowRight':
            state.savePromptButtonIndex = (state.savePromptButtonIndex + 1) % state.savePromptButtons.length;
            ui.updateSavePromptButtonSelection();
            break;
        case 'Enter':
            if (state.savePromptButtonIndex !== -1) state.savePromptButtons[state.savePromptButtonIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}

function handlePowerMenuKeys(event) {
    switch (event.key) {
        case 'ArrowLeft':
            state.powerMenuButtonIndex = (state.powerMenuButtonIndex - 1 + state.powerMenuButtons.length) % state.powerMenuButtons.length;
            ui.updatePowerMenuSelection();
            break;
        case 'ArrowRight':
            state.powerMenuButtonIndex = (state.powerMenuButtonIndex + 1) % state.powerMenuButtons.length;
            ui.updatePowerMenuSelection();
            break;
        case 'Enter':
            if (state.powerMenuButtonIndex !== -1) {
                state.powerMenuButtons[state.powerMenuButtonIndex]?.click();
            }
            break;
        default: return;
    }
    event.preventDefault();
}

function handleLobbyMenuKeys(event) {
    if (state.lobbyGames.length === 0) return;
    switch (event.key) {
        case 'ArrowUp':
            state.lobbySelectionIndex = (state.lobbySelectionIndex - 1 + state.lobbyGames.length) % state.lobbyGames.length;
            ui.updateLobbySelection();
            break;
        case 'ArrowDown':
            state.lobbySelectionIndex = (state.lobbySelectionIndex + 1) % state.lobbyGames.length;
            ui.updateLobbySelection();
            break;
        case 'Enter':
            if (state.lobbySelectionIndex !== -1) dom.lobbyListUl.children[state.lobbySelectionIndex]?.click();
            break;
        default: return;
    }
    event.preventDefault();
}